const express = require('express');
const cors = require('cors');
const helmet = require('helmet');
const rateLimit = require('express-rate-limit');
const path = require('path');
const compression = require('compression');
require('dotenv').config();

// Import routes
const authRoutes = require('./routes/auth');
const productRoutes = require('./routes/products');
const orderRoutes = require('./routes/orders');
const userRoutes = require('./routes/users');
const categoryRoutes = require('./routes/categories');
const paymentRoutes = require('./routes/payments');

// Import database initialization
const { initDatabase } = require('./database/init');
const { displayNetworkInfo } = require('./utils/network');
const { errorHandler, dbCleanup, requestTimeout, rateLimitErrorHandler } = require('./middleware/errorHandler');

const app = express();
const PORT = process.env.PORT || 3001;

// Enhanced rate limiting with different limits for different endpoints
// More permissive limits to prevent testing issues
const generalLimiter = rateLimit({
  windowMs: 1 * 60 * 1000, // 1 minute
  max: 1000, // much higher limit to allow extensive testing
  message: 'Too many requests from this IP, please try again later.',
  standardHeaders: true,
  legacyHeaders: false,
  skip: (req) => {
    // Skip rate limiting for health checks and docs
    return req.path === '/api/health' || req.path === '/api/docs';
  }
});

const strictLimiter = rateLimit({
  windowMs: 1 * 60 * 1000, // 1 minute
  max: 500, // higher limit for auth operations
  message: 'Too many requests from this IP, please try again later.',
  standardHeaders: true,
  legacyHeaders: false,
});

// Middleware
app.use(
  helmet({
    crossOriginResourcePolicy: false,
  })
);

// Compression middleware for better performance
app.use(compression());

// Database cleanup middleware
app.use(dbCleanup);

// Request timeout middleware
app.use(requestTimeout(30000));

// Apply rate limiting
app.use(generalLimiter);

// Enhanced CORS middleware - handle all CORS issues
app.use((req, res, next) => {
  const origin = req.headers.origin;
  const isDevelopment = process.env.NODE_ENV !== 'production';
  
  if (isDevelopment) {
    // In development, allow all origins
    res.header('Access-Control-Allow-Origin', origin || '*');
  } else {
    // In production, check allowed origins
    const allowedOrigins = [
      'http://localhost:3000',
      'http://127.0.0.1:3000',
      'http://192.168.0.15:3000'
    ];
    
    if (!origin || allowedOrigins.includes(origin)) {
      res.header('Access-Control-Allow-Origin', origin || 'http://localhost:3000');
    }
  }
  
  res.header('Access-Control-Allow-Credentials', 'true');
  res.header('Access-Control-Allow-Methods', 'GET,POST,PUT,DELETE,PATCH,OPTIONS,HEAD');
  res.header('Access-Control-Allow-Headers', [
    'Origin',
    'X-Requested-With', 
    'Content-Type',
    'Accept',
    'Authorization',
    'Cache-Control',
    'X-File-Name'
  ].join(', '));
  
  // Handle preflight requests
  if (req.method === 'OPTIONS') {
    res.header('Access-Control-Max-Age', '86400'); // 24 hours
    return res.status(204).send();
  }
  
  next();
});

// Handle preflight requests first
app.options('*', cors({
  origin: true,
  credentials: true,
  methods: ['GET', 'POST', 'PUT', 'DELETE', 'PATCH', 'OPTIONS'],
  allowedHeaders: ['Content-Type', 'Authorization', 'X-Requested-With', 'Accept', 'Origin'],
}));

// CORS configuration - more permissive for development
const isDevelopment = process.env.NODE_ENV !== 'production';

if (isDevelopment) {
  // In development, allow all origins for easier testing
  app.use(cors({
    origin: true, // Allow all origins in development
    credentials: true,
    methods: ['GET', 'POST', 'PUT', 'DELETE', 'PATCH', 'OPTIONS'],
    allowedHeaders: ['Content-Type', 'Authorization', 'X-Requested-With', 'Accept', 'Origin']
  }));
  console.log('CORS: Development mode - allowing all origins');
} else {
  // In production, use strict CORS
  const allowedOrigins = [
    'http://localhost:3000',
    'http://192.168.1.7:3000',
    'http://localhost:3001',
    'http://192.168.1.7:3001',
    'http://192.168.1.7:3002',
    'http://localhost:3002',
    // Allow any localhost or local IP for development
    /^http:\/\/localhost:\d+$/,
    /^http:\/\/192\.168\.\d+\.\d+:\d+$/,
    /^http:\/\/127\.0\.0\.1:\d+$/
  ];

  app.use(cors({
    origin: function(origin, callback) {
      // allow requests with no origin (like mobile apps, curl, etc.)
      if (!origin) return callback(null, true);
      
      // Check if origin matches any allowed pattern
      const isAllowed = allowedOrigins.some(allowed => {
        if (typeof allowed === 'string') {
          return origin === allowed;
        } else if (allowed instanceof RegExp) {
          return allowed.test(origin);
        }
        return false;
      });
      
      if (isAllowed) {
        return callback(null, true);
      } else {
        console.log('CORS blocked origin:', origin);
        return callback(new Error('Not allowed by CORS'));
      }
    },
    credentials: true,
    methods: ['GET', 'POST', 'PUT', 'DELETE', 'PATCH', 'OPTIONS'],
    allowedHeaders: ['Content-Type', 'Authorization', 'X-Requested-With']
  }));
  console.log('CORS: Production mode - using strict origin checking');
}

// Increase payload limits for file uploads
app.use(express.json({ limit: '50mb' }));
app.use(express.urlencoded({ extended: true, limit: '50mb' }));

// Serve static files (uploads) with caching
app.use('/uploads', express.static(path.join(__dirname, 'uploads'), {
  maxAge: '1d', // Cache static files for 1 day
  etag: true,
  lastModified: true
}));

// Routes with appropriate rate limiting
app.use('/api/auth', authRoutes); // Remove strict limiter from all auth routes
app.use('/api/products', productRoutes);
app.use('/api/categories', categoryRoutes);
app.use('/api/orders', orderRoutes);
app.use('/api/users', userRoutes); // Remove strict limiter from all user routes
app.use('/api/payments', paymentRoutes);

// Health check endpoint
app.get('/api/health', (req, res) => {
  res.json({ 
    status: 'OK', 
    timestamp: new Date().toISOString(),
    uptime: process.uptime()
  });
});

// API documentation endpoint
app.get('/api/docs', (req, res) => {
  res.json({
    message: 'Usasya API Documentation',
    version: '1.0.0',
    endpoints: {
      auth: {
        'POST /api/auth/register': 'Register new client user',
        'POST /api/auth/login': 'Login user',
        'GET /api/auth/me': 'Get current user info',
        'POST /api/auth/refresh': 'Refresh auth token',
        'POST /api/auth/logout': 'Logout user'
      },
      users: {
        'GET /api/users': 'Get all users (staff/admin only)',
        'GET /api/users/:id': 'Get user by ID',
        'POST /api/users': 'Create new user (admin only)',
        'PUT /api/users/:id': 'Update user',
        'PATCH /api/users/profile': 'Update own profile',
        'PATCH /api/users/:id/password': 'Change password',
        'PATCH /api/users/:id/status': 'Update user status (staff/admin only)',
        'DELETE /api/users/:id': 'Delete user (admin only)',
        'GET /api/users/stats/overview': 'Get user statistics (staff/admin only)'
      },
      categories: {
        'GET /api/categories': 'Get all categories',
        'GET /api/categories/:id': 'Get category by ID',
        'POST /api/categories': 'Create category (staff/admin only)',
        'PUT /api/categories/:id': 'Update category (staff/admin only)',
        'DELETE /api/categories/:id': 'Delete category (staff/admin only)'
      },
      products: {
        'GET /api/products': 'Get all products',
        'GET /api/products/:id': 'Get product by ID',
        'POST /api/products': 'Create product (staff/admin only)',
        'PUT /api/products/:id': 'Update product (staff/admin only)',
        'DELETE /api/products/:id': 'Delete product (staff/admin only)'
      },
      orders: {
        'GET /api/orders': 'Get all orders',
        'GET /api/orders/:id': 'Get order by ID',
        'POST /api/orders': 'Create new order',
        'PUT /api/orders/:id': 'Update order',
        'DELETE /api/orders/:id': 'Delete order',
        'GET /api/orders/manual': 'Get all manual orders (staff/admin only)',
        'POST /api/orders/manual': 'Create manual order (staff/admin only)',
        'PUT /api/orders/manual/:id': 'Update manual order (staff/admin only)',
        'DELETE /api/orders/manual/:id': 'Delete manual order (staff/admin only)'
      }
    }
  });
});

// 404 handler
app.use('*', (req, res) => {
  res.status(404).json({ error: 'Route not found' });
});

// Rate limiting error handler
app.use(rateLimitErrorHandler);

// Global error handler
app.use(errorHandler);

// Initialize database and start server
async function startServer() {
  try {
    await initDatabase();
    
    app.listen(PORT, '0.0.0.0', () => {
      console.log(`Server running on port ${PORT}`);
      console.log(`Environment: ${process.env.NODE_ENV || 'development'}`);
      
      // Display network access information
      displayNetworkInfo(PORT);
    });
  } catch (error) {
    console.error('Failed to start server:', error);
    process.exit(1);
  }
}

startServer();

module.exports = app;
